<?php
/**
 * CLAWS Shortcodes Class
 * Handles all shortcodes for the CLAWS app
 */

if (!defined('ABSPATH')) {
    exit;
}

class CLAWS_Shortcodes {
    
    private static $instance = null;
    
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        add_shortcode('claws_app', array($this, 'render_app'));
        add_shortcode('claws_colony_tracker', array($this, 'render_colony_tracker'));
        add_shortcode('claws_intake_discharge', array($this, 'render_intake_discharge'));
        add_shortcode('claws_neonate_care', array($this, 'render_neonate_care'));
        add_shortcode('Admin_Reports', array($this, 'render_admin_reports'));
        add_shortcode('claws_summary', array($this, 'render_summary'));
        
        // Enqueue scripts when shortcode is used
        add_action('wp_enqueue_scripts', array($this, 'maybe_enqueue_scripts'));
    }
    
    /**
     * Main CLAWS app shortcode
     */
    public function render_app($atts) {
        $atts = shortcode_atts(array(
            'height' => '100vh',
            'show_login_prompt' => 'true'
        ), $atts, 'claws_app');
        
        // AGGRESSIVE CACHE BUSTING - Force fresh template load
        if (!headers_sent()) {
            header('Cache-Control: no-cache, no-store, must-revalidate, max-age=0');
            header('Pragma: no-cache');
            header('Expires: 0');
        }
        
        // Ensure scripts are loaded
        $this->enqueue_app_scripts();
        
        ob_start();
        include CLAWS_PLUGIN_DIR . 'templates/app-container.php';
        $output = ob_get_clean();
        
        // Add timestamp comment to force cache bypass
        $output = "<!-- PURRALYTICS v" . CLAWS_VERSION . " loaded at " . date('Y-m-d H:i:s') . " -->\n" . $output;
        
        return $output;
    }
    
    /**
     * Colony tracker only shortcode
     */
    public function render_colony_tracker($atts) {
        $atts = shortcode_atts(array(
            'show_map' => 'true',
            'show_statistics' => 'true',
            'show_pie_chart' => 'true'
        ), $atts, 'claws_colony_tracker');
        
        $this->enqueue_app_scripts();
        
        ob_start();
        ?>
        <div id="claws-colony-tracker" class="claws-standalone-module">
            <div class="colony-tracker-content">
                <!-- Colony tracker will be loaded here by JavaScript -->
                <div class="loading-message" style="text-align: center; padding: 2rem; color: #6b7280;">
                    <p>Loading Colony Tracker...</p>
                </div>
            </div>
        </div>
        <script>
        document.addEventListener('DOMContentLoaded', function() {
            if (window.CLAWSApp && window.CLAWSApp.initColonyTracker) {
                window.CLAWSApp.initColonyTracker('claws-colony-tracker', <?php echo wp_json_encode($atts); ?>);
            }
        });
        </script>
        <?php
        return ob_get_clean();
    }
    
    /**
     * Intake/Discharge only shortcode
     */
    public function render_intake_discharge($atts) {
        $atts = shortcode_atts(array(
            'default_tab' => 'intake'
        ), $atts, 'claws_intake_discharge');
        
        $this->enqueue_app_scripts();
        
        ob_start();
        ?>
        <div id="claws-intake-discharge" class="claws-standalone-module">
            <div class="intake-discharge-content">
                <!-- Intake/Discharge will be loaded here by JavaScript -->
                <div class="loading-message" style="text-align: center; padding: 2rem; color: #6b7280;">
                    <p>Loading Intake & Discharge...</p>
                </div>
            </div>
        </div>
        <script>
        document.addEventListener('DOMContentLoaded', function() {
            if (window.CLAWSApp && window.CLAWSApp.initIntakeDischarge) {
                window.CLAWSApp.initIntakeDischarge('claws-intake-discharge', <?php echo wp_json_encode($atts); ?>);
            }
        });
        </script>
        <?php
        return ob_get_clean();
    }
    
    /**
     * Neonate care only shortcode
     */
    public function render_neonate_care($atts) {
        $atts = shortcode_atts(array(
            'default_age_range' => '0-1-weeks'
        ), $atts, 'claws_neonate_care');
        
        $this->enqueue_app_scripts();
        
        ob_start();
        ?>
        <div id="claws-neonate-care" class="claws-standalone-module">
            <div class="neonate-care-content">
                <!-- Neonate care will be loaded here by JavaScript -->
                <div class="loading-message" style="text-align: center; padding: 2rem; color: #6b7280;">
                    <p>Loading NeoNate Care...</p>
                </div>
            </div>
        </div>
        <script>
        document.addEventListener('DOMContentLoaded', function() {
            if (window.CLAWSApp && window.CLAWSApp.initNeonateCare) {
                window.CLAWSApp.initNeonateCare('claws-neonate-care', <?php echo wp_json_encode($atts); ?>);
            }
        });
        </script>
        <?php
        return ob_get_clean();
    }
    
    /**
     * Count posts for a specific user (security fix)
     */
    private function count_user_posts($post_type, $user_id) {
        $posts = get_posts(array(
            'post_type' => $post_type,
            'author' => $user_id,
            'post_status' => 'publish',
            'numberposts' => -1,
            'fields' => 'ids' // Only get IDs for performance
        ));
        return count($posts);
    }
    
    /**
     * Count posts accessible to user based on role
     */
    private function count_accessible_posts($post_type, $user_id) {
        // Check if user is contributor-only
        $current_user = wp_get_current_user();
        $is_contributor_only = in_array('contributor', $current_user->roles) &&
                               !in_array('author', $current_user->roles) &&
                               !in_array('administrator', $current_user->roles);
        
        if ($is_contributor_only) {
            // Contributors only see their own records
            $posts = get_posts(array(
                'post_type' => $post_type,
                'author' => $user_id,
                'post_status' => 'publish',
                'numberposts' => -1,
                'fields' => 'ids'
            ));
            return count($posts);
        } else {
            // Authors and admins see all records
            return wp_count_posts($post_type)->publish;
        }
    }
    
    /**
     * Get recent activity filtered by user permissions
     */
    private function get_recent_activity($user_id = null) {
        // Determine query args based on user role
        $query_args = array(
            'post_type' => array('claws_colony', 'claws_intake', 'claws_discharge'),
            'numberposts' => 10,
            'post_status' => 'publish',
            'orderby' => 'date',
            'order' => 'DESC'
        );
        
        // If user is contributor-only, filter to their posts
        if ($user_id) {
            $current_user = wp_get_current_user();
            $is_contributor_only = in_array('contributor', $current_user->roles) &&
                                   !in_array('author', $current_user->roles) &&
                                   !in_array('administrator', $current_user->roles);
            
            if ($is_contributor_only) {
                $query_args['author'] = $user_id;
            }
        }
        
        $recent_posts = get_posts($query_args);
        
        $activity = array();
        foreach ($recent_posts as $post) {
            $user = get_userdata($post->post_author);
            $activity[] = array(
                'type' => str_replace('claws_', '', $post->post_type),
                'title' => $post->post_title,
                'author' => $user ? $user->display_name : 'Unknown',
                'date' => $post->post_date
            );
        }
        
        return $activity;
    }

    /**
     * Check if we need to enqueue scripts
     */
    public function maybe_enqueue_scripts() {
        global $post;
        
        if (is_a($post, 'WP_Post')) {
            $content = $post->post_content;
            
            // Check for any CLAWS shortcodes
            if (has_shortcode($content, 'claws_app') || 
                has_shortcode($content, 'claws_colony_tracker') || 
                has_shortcode($content, 'claws_intake_discharge') || 
                has_shortcode($content, 'claws_neonate_care') ||
                has_shortcode($content, 'Admin_Reports') ||
                has_shortcode($content, 'claws_summary')) {
                
                $this->enqueue_app_scripts();
            }
        }
    }
    
    /**
     * Enqueue all necessary scripts and styles
     */
    private function enqueue_app_scripts() {
        // External libraries
        wp_enqueue_script('leaflet', 'https://unpkg.com/leaflet@1.9.4/dist/leaflet.js', array(), '1.9.4', true);
        wp_enqueue_style('leaflet', 'https://unpkg.com/leaflet@1.9.4/dist/leaflet.css', array(), '1.9.4');
        
        wp_enqueue_script('chartjs', 'https://cdn.jsdelivr.net/npm/chart.js', array(), '4.4.0', true);
        wp_enqueue_script('chartjs-adapter-date-fns', 'https://cdn.jsdelivr.net/npm/chartjs-adapter-date-fns/dist/chartjs-adapter-date-fns.bundle.min.js', array('chartjs'), '3.0.0', true);
        wp_enqueue_script('lucide', 'https://unpkg.com/lucide@latest/dist/umd/lucide.js', array(), 'latest', true);
        
        // CLAWS app files - cache busting using version number
        $cache_buster = CLAWS_VERSION;
        
        // Enqueue modular JavaScript files in correct order
        wp_enqueue_script('claws-constants', CLAWS_PLUGIN_URL . 'assets/js/claws-constants.js', 
            array(), $cache_buster, true);
        
        wp_enqueue_script('claws-core', CLAWS_PLUGIN_URL . 'assets/js/claws-core.js', 
            array('jquery', 'claws-constants'), $cache_buster, true);
            
        wp_enqueue_script('claws-ui', CLAWS_PLUGIN_URL . 'assets/js/claws-ui.js', 
            array('claws-core'), $cache_buster, true);
            
        wp_enqueue_script('claws-forms', CLAWS_PLUGIN_URL . 'assets/js/claws-forms.js', 
            array('claws-core'), $cache_buster, true);
            
        wp_enqueue_script('claws-maps', CLAWS_PLUGIN_URL . 'assets/js/claws-maps.js', 
            array('claws-core', 'leaflet'), $cache_buster, true);
            
        wp_enqueue_script('claws-charts', CLAWS_PLUGIN_URL . 'assets/js/claws-charts.js', 
            array('claws-core', 'chartjs'), $cache_buster, true);
            
        wp_enqueue_script('claws-colony', CLAWS_PLUGIN_URL . 'assets/js/claws-colony.js', 
            array('claws-core', 'claws-ui', 'claws-forms', 'claws-maps', 'claws-charts'), $cache_buster, true);
        
        // EMERGENCY FIX: Force update colony modal to bypass caching
        wp_enqueue_script('claws-modal-fix', CLAWS_PLUGIN_URL . 'assets/js/fix-colony-modal.js', 
            array('claws-colony'), $cache_buster, true);
        
        wp_enqueue_style('claws-main-css', CLAWS_PLUGIN_URL . 'assets/css/claws-app.css', 
            array('leaflet'), $cache_buster);
        
        // Localize script with WordPress data
        $current_user = wp_get_current_user();
        $user_roles = is_user_logged_in() ? $current_user->roles : array();
        $user_id = get_current_user_id();
        
        // Count only user's own records for contributors
        $is_contributor_only = in_array('contributor', $user_roles) &&
                               !in_array('author', $user_roles) &&
                               !in_array('administrator', $user_roles);
        
        wp_localize_script('claws-core', 'clawsData', array(
            'apiUrl' => rest_url('claws/v1/'),
            'nonce' => wp_create_nonce('wp_rest'),
            'isLoggedIn' => is_user_logged_in(),
            'loginUrl' => wp_login_url(get_permalink()),
            'currentUser' => is_user_logged_in() ? $current_user->display_name : '',
            'currentUserEmail' => is_user_logged_in() ? $current_user->user_email : '',
            'userId' => $user_id,
            'userLogin' => is_user_logged_in() ? $current_user->user_login : '',
            'userRoles' => $user_roles,
            'userRole' => !empty($user_roles) ? $user_roles[0] : 'subscriber',
            'isContributor' => in_array('contributor', $user_roles),
            'isAuthor' => in_array('author', $user_roles),
            'isAdmin' => in_array('administrator', $user_roles),
            'isContributorOnly' => $is_contributor_only,
            'canEdit' => current_user_can('edit_posts'),
            'canCreateColonies' => (in_array('contributor', $user_roles) || in_array('author', $user_roles) || in_array('administrator', $user_roles)),
            'canEditOthers' => current_user_can('edit_others_posts'),
            // Count only user's own records for contributors
            'total_intakes' => $is_contributor_only ? $this->count_user_posts('claws_intake', $user_id) : wp_count_posts('claws_intake')->publish,
            'total_discharges' => $is_contributor_only ? $this->count_user_posts('claws_discharge', $user_id) : wp_count_posts('claws_discharge')->publish,
            'recent_activity' => $this->get_recent_activity($user_id),
            'pluginUrl' => CLAWS_PLUGIN_URL,
            'ajaxUrl' => admin_url('admin-ajax.php'),
            // Custom branding and settings
            'organizationName' => get_option('claws_organization_name', 'CLAWS'),
            'organizationTagline' => get_option('claws_organization_tagline', 'Compassionate care for every cat'),
            'defaultLatitude' => get_option('claws_default_latitude', 39.8283),
            'defaultLongitude' => get_option('claws_default_longitude', -98.5795),
            'defaultLocation' => get_option('claws_default_location', 'United States'),
            // Color scheme settings
            'colorSchemeType' => get_option('claws_color_scheme_type', 'gradient'),
            'primaryColor' => get_option('claws_primary_color', '#C1292E'),
            'secondaryColor' => get_option('claws_secondary_color', '#00274D'),
            // Logo settings
            'organizationLogo' => $this->get_organization_logo_url()
        ));
        
        // Add inline CSS for immediate styling
        $inline_css = '
        .claws-app-wrapper { 
            font-family: system-ui, -apple-system, sans-serif; 
        }
        .loading-message { 
            padding: 2rem; 
            text-align: center; 
            color: #6b7280; 
        }
        .claws-standalone-module { 
            max-width: 28rem; 
            margin: 0 auto; 
            background: #f9fafb; 
            border-radius: 1rem; 
            overflow: hidden; 
        }
        @media (min-width: 1024px) {
            .claws-standalone-module {
                max-width: 800px;
            }
        }
        ';
        wp_add_inline_style('claws-main-css', $inline_css);
    }
    
    /**
     * Admin Reports shortcode for frontend display
     */
    public function render_admin_reports($atts) {
        $atts = shortcode_atts(array(
            'show_summary' => 'true',
            'show_charts' => 'true',
            'title' => 'How We\'re Doing'
        ), $atts, 'Admin_Reports');
        
        // Ensure scripts are loaded
        $this->enqueue_app_scripts();
        
        // Get data for summary stats
        $colonies = CLAWS_Database::getInstance()->get_colonies();
        $total_cats = array_sum(array_column($colonies, 'cats'));
        $total_intakes = wp_count_posts('claws_intake')->publish;
        $total_discharges = wp_count_posts('claws_discharge')->publish;
        
        ob_start();
        ?>
        <div id="claws-admin-reports" class="claws-admin-reports-wrapper" style="max-width: 1200px; margin: 0 auto; font-family: system-ui, -apple-system, sans-serif;">
            
            <?php if ($atts['show_summary'] === 'true'): ?>
            <!-- Title Section -->
            <div style="text-align: center; margin-bottom: 2rem;">
                <h2 style="font-size: 2rem; font-weight: bold; color: #1f2937; margin: 0 0 0.5rem;"><?php echo esc_html($atts['title']); ?></h2>
                <p style="color: #6b7280; font-size: 1.125rem;"><?php echo esc_html(get_option('claws_organization_name', 'CLAWS')); ?> Organization Impact Summary</p>
            </div>
            
            <!-- Summary Stats -->
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 1rem; margin-bottom: 3rem;">
                <div style="background: linear-gradient(135deg, #3b82f6, #1d4ed8); color: white; padding: 1.5rem; border-radius: 1rem; text-align: center; box-shadow: 0 10px 15px -3px rgba(59, 130, 246, 0.4);">
                    <div style="font-size: 2.5rem; font-weight: bold; margin-bottom: 0.5rem;"><?php echo count($colonies); ?></div>
                    <div style="font-size: 1rem; opacity: 0.9;">Total Colonies</div>
                </div>
                <div style="background: linear-gradient(135deg, #ef4444, #dc2626); color: white; padding: 1.5rem; border-radius: 1rem; text-align: center; box-shadow: 0 10px 15px -3px rgba(239, 68, 68, 0.4);">
                    <div style="font-size: 2.5rem; font-weight: bold; margin-bottom: 0.5rem;"><?php echo $total_cats; ?></div>
                    <div style="font-size: 1rem; opacity: 0.9;">Total Cats</div>
                </div>
                <div style="background: linear-gradient(135deg, #10b981, #059669); color: white; padding: 1.5rem; border-radius: 1rem; text-align: center; box-shadow: 0 10px 15px -3px rgba(16, 185, 129, 0.4);">
                    <div style="font-size: 2.5rem; font-weight: bold; margin-bottom: 0.5rem;"><?php echo $total_intakes; ?></div>
                    <div style="font-size: 1rem; opacity: 0.9;">Total Intakes</div>
                </div>
                <div style="background: linear-gradient(135deg, #f59e0b, #d97706); color: white; padding: 1.5rem; border-radius: 1rem; text-align: center; box-shadow: 0 10px 15px -3px rgba(245, 158, 11, 0.4);">
                    <div style="font-size: 2.5rem; font-weight: bold; margin-bottom: 0.5rem;"><?php echo $total_discharges; ?></div>
                    <div style="font-size: 1rem; opacity: 0.9;">Total Outcomes</div>
                </div>
            </div>
            <?php endif; ?>
            
            <?php if ($atts['show_charts'] === 'true'): ?>
            <!-- Analytics Charts -->
            <div style="margin-bottom: 2rem;">
                <h3 style="font-size: 1.5rem; font-weight: bold; color: #1f2937; text-align: center; margin: 0 0 2rem;">Analytics Overview</h3>
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(350px, 1fr)); gap: 2rem;">
                    
                    <!-- Intake Type Chart -->
                    <div style="background: white; border: 1px solid #e5e7eb; border-radius: 1rem; padding: 1.5rem; box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);">
                        <h4 style="font-size: 1.125rem; font-weight: 600; color: #1f2937; text-align: center; margin: 0 0 1rem;">Intakes by Type</h4>
                        <div style="position: relative; height: 300px;">
                            <canvas id="frontend-intake-type-chart" style="max-height: 300px;"></canvas>
                        </div>
                    </div>
                    
                    <!-- Intake Gender Chart -->
                    <div style="background: white; border: 1px solid #e5e7eb; border-radius: 1rem; padding: 1.5rem; box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);">
                        <h4 style="font-size: 1.125rem; font-weight: 600; color: #1f2937; text-align: center; margin: 0 0 1rem;">Intakes by Gender</h4>
                        <div style="position: relative; height: 300px;">
                            <canvas id="frontend-intake-gender-chart" style="max-height: 300px;"></canvas>
                        </div>
                    </div>
                    
                    <!-- Intake Age Chart -->
                    <div style="background: white; border: 1px solid #e5e7eb; border-radius: 1rem; padding: 1.5rem; box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);">
                        <h4 style="font-size: 1.125rem; font-weight: 600; color: #1f2937; text-align: center; margin: 0 0 1rem;">Intakes by Age</h4>
                        <div style="position: relative; height: 300px;">
                            <canvas id="frontend-intake-age-chart" style="max-height: 300px;"></canvas>
                        </div>
                    </div>
                    
                    <!-- Discharge Outcomes Chart -->
                    <div style="background: white; border: 1px solid #e5e7eb; border-radius: 1rem; padding: 1.5rem; box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);">
                        <h4 style="font-size: 1.125rem; font-weight: 600; color: #1f2937; text-align: center; margin: 0 0 1rem;">Discharge Outcomes</h4>
                        <div style="position: relative; height: 300px;">
                            <canvas id="frontend-discharge-outcome-chart" style="max-height: 300px;"></canvas>
                        </div>
                    </div>
                    
                </div>
            </div>
            <?php endif; ?>
            
            <!-- Database Management - Only show in WordPress admin, not on frontend -->
            <?php if (current_user_can('manage_options') && is_admin()): ?>
            <div style="margin-bottom: 2rem;">
                <h3 style="font-size: 1.5rem; font-weight: bold; color: #1f2937; text-align: center; margin: 0 0 2rem;">Database Management (Admin Only)</h3>
                <div style="background: white; border: 1px solid #e5e7eb; border-radius: 1rem; padding: 1.5rem; box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);">
                    
                    <!-- Feeding Records Cleanup -->
                    <div style="background: #fef3c7; border: 1px solid #f59e0b; border-radius: 0.5rem; padding: 1rem; margin-bottom: 1.5rem;">
                        <h4 style="font-size: 1.125rem; font-weight: 600; color: #92400e; margin: 0 0 0.5rem;">⚠️ Feeding Records Cleanup</h4>
                        <p style="color: #92400e; font-size: 0.875rem; margin: 0 0 1rem;">Delete feeding records by date range to keep the database manageable. Records for active kittens are protected by default.</p>
                        
                        <div style="display: grid; grid-template-columns: 1fr 1fr 1fr auto; gap: 1rem; align-items: end; margin-bottom: 1rem;">
                            <div>
                                <label style="display: block; font-weight: 600; color: #374151; margin-bottom: 0.25rem; font-size: 0.875rem;">Start Date</label>
                                <input type="date" id="bulk-delete-start-date" style="width: 100%; padding: 0.5rem; border: 1px solid #d1d5db; border-radius: 0.375rem; font-size: 0.875rem;">
                            </div>
                            <div>
                                <label style="display: block; font-weight: 600; color: #374151; margin-bottom: 0.25rem; font-size: 0.875rem;">End Date</label>
                                <input type="date" id="bulk-delete-end-date" style="width: 100%; padding: 0.5rem; border: 1px solid #d1d5db; border-radius: 0.375rem; font-size: 0.875rem;">
                            </div>
                            <div>
                                <label style="display: block; font-weight: 600; color: #374151; margin-bottom: 0.25rem; font-size: 0.875rem;">
                                    <input type="checkbox" id="protect-active-kittens" checked style="margin-right: 0.5rem;">
                                    Protect Active Kittens
                                </label>
                                <p style="color: #6b7280; font-size: 0.75rem; margin: 0;">Only delete records for discharged kittens</p>
                            </div>
                            <div>
                                <button onclick="bulkDeleteFeedingRecords()" style="background: #dc2626; color: white; border: none; border-radius: 0.375rem; padding: 0.5rem 1rem; font-size: 0.875rem; font-weight: 600; cursor: pointer; transition: background-color 0.2s;" onmouseover="this.style.backgroundColor='#b91c1c'" onmouseout="this.style.backgroundColor='#dc2626'">
                                    🗑️ Delete Records
                                </button>
                            </div>
                        </div>
                        
                        <div id="bulk-delete-result" style="display: none; padding: 0.75rem; border-radius: 0.375rem; font-size: 0.875rem; font-weight: 500;"></div>
                    </div>
                </div>
            </div>
            <?php endif; ?>
            
            <!-- Footer -->
            <div style="text-align: center; padding: 2rem 0; border-top: 1px solid #e5e7eb; margin-top: 2rem;">
                <p style="color: #6b7280; font-size: 0.875rem; margin: 0;">
                    Data updated in real-time • Powered by <?php echo esc_html(get_option('claws_organization_name', 'CLAWS')); ?> Management System
                </p>
            </div>
            
        </div>
        
        <script>
        document.addEventListener('DOMContentLoaded', function() {
            loadFrontendAggregateCharts();
        });
        
        <?php if (current_user_can('manage_options') && is_admin()): ?>
        // Bulk delete feeding records function (admin only)
        window.bulkDeleteFeedingRecords = async function() {
            const startDate = document.getElementById('bulk-delete-start-date').value;
            const endDate = document.getElementById('bulk-delete-end-date').value;
            const protectActive = document.getElementById('protect-active-kittens').checked;
            const resultDiv = document.getElementById('bulk-delete-result');
            
            // Validation
            if (!startDate || !endDate) {
                window.showBulkDeleteResult('Please select both start and end dates.', 'error');
                return;
            }
            
            if (new Date(startDate) > new Date(endDate)) {
                window.showBulkDeleteResult('Start date must be before or equal to end date.', 'error');
                return;
            }
            
            // Confirmation
            const protectionText = protectActive ? ' (records for active kittens will be protected)' : ' (ALL records in range will be deleted)';
            if (!confirm(`Are you sure you want to delete feeding records from ${startDate} to ${endDate}?${protectionText}\n\nThis action cannot be undone.`)) {
                return;
            }
            
            try {
                resultDiv.style.display = 'block';
                resultDiv.innerHTML = '⏳ Deleting records...';
                resultDiv.style.background = '#dbeafe';
                resultDiv.style.color = '#1e40af';
                
                const response = await fetch('<?php echo rest_url('claws/v1/feeding/bulk-delete'); ?>', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-WP-Nonce': '<?php echo wp_create_nonce('wp_rest'); ?>'
                    },
                    body: JSON.stringify({
                        start_date: startDate,
                        end_date: endDate,
                        exclude_active_kittens: protectActive
                    })
                });
                
                const result = await response.json();
                
                if (response.ok && result.success) {
                    window.showBulkDeleteResult(result.message, 'success');
                } else {
                    window.showBulkDeleteResult(result.message || 'Failed to delete records.', 'error');
                }
                
            } catch (error) {
                console.error('Bulk delete error:', error);
                window.showBulkDeleteResult('An error occurred while deleting records.', 'error');
            }
        };
        
        window.showBulkDeleteResult = function(message, type) {
            const resultDiv = document.getElementById('bulk-delete-result');
            resultDiv.style.display = 'block';
            resultDiv.innerHTML = message;
            
            if (type === 'success') {
                resultDiv.style.background = '#d1fae5';
                resultDiv.style.color = '#065f46';
            } else if (type === 'error') {
                resultDiv.style.background = '#fee2e2';
                resultDiv.style.color = '#991b1b';
            }
            
            // Auto-hide after 10 seconds
            setTimeout(() => {
                resultDiv.style.display = 'none';
            }, 10000);
        };
        <?php endif; ?>
        
        function loadFrontendAggregateCharts() {
            // Load all intake data
            fetch('<?php echo rest_url('claws/v1/admin/intakes'); ?>', {
                credentials: 'include',
                headers: { 'X-WP-Nonce': '<?php echo wp_create_nonce('wp_rest'); ?>' }
            })
            .then(response => response.json())
            .then(intakes => {
                createFrontendIntakeTypeChart(intakes);
                createFrontendIntakeGenderChart(intakes);
                createFrontendIntakeAgeChart(intakes);
            })
            .catch(error => console.error('Error loading intake data:', error));
            
            // Load all discharge data
            fetch('<?php echo rest_url('claws/v1/admin/discharges'); ?>', {
                credentials: 'include',
                headers: { 'X-WP-Nonce': '<?php echo wp_create_nonce('wp_rest'); ?>' }
            })
            .then(response => response.json())
            .then(discharges => {
                createFrontendDischargeOutcomeChart(discharges);
            })
            .catch(error => console.error('Error loading discharge data:', error));
        }
        
        function createFrontendIntakeTypeChart(intakes) {
            const canvas = document.getElementById('frontend-intake-type-chart');
            if (!canvas) return;
            
            const typeCounts = {};
            intakes.forEach(intake => {
                const type = intake.type || 'Unknown';
                typeCounts[type] = (typeCounts[type] || 0) + 1;
            });
            
            const colors = ['#ef4444', '#f97316', '#eab308', '#22c55e', '#06b6d4', '#3b82f6', '#8b5cf6', '#ec4899'];
            const chartData = Object.entries(typeCounts).map(([type, count], index) => ({
                label: type,
                data: count,
                backgroundColor: colors[index % colors.length]
            }));
            
            new Chart(canvas.getContext('2d'), {
                type: 'pie',
                data: {
                    labels: chartData.map(d => d.label),
                    datasets: [{
                        data: chartData.map(d => d.data),
                        backgroundColor: chartData.map(d => d.backgroundColor)
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false
                }
            });
        }
        
        function createFrontendIntakeGenderChart(intakes) {
            const canvas = document.getElementById('frontend-intake-gender-chart');
            if (!canvas) return;
            
            const genderCounts = {};
            intakes.forEach(intake => {
                const gender = intake.gender || 'Unknown';
                genderCounts[gender] = (genderCounts[gender] || 0) + 1;
            });
            
            const colors = ['#ef4444', '#f97316', '#eab308', '#22c55e', '#06b6d4', '#3b82f6', '#8b5cf6', '#ec4899'];
            const chartData = Object.entries(genderCounts).map(([gender, count], index) => ({
                label: gender,
                data: count,
                backgroundColor: colors[index % colors.length]
            }));
            
            new Chart(canvas.getContext('2d'), {
                type: 'pie',
                data: {
                    labels: chartData.map(d => d.label),
                    datasets: [{
                        data: chartData.map(d => d.data),
                        backgroundColor: chartData.map(d => d.backgroundColor)
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false
                }
            });
        }
        
        function createFrontendIntakeAgeChart(intakes) {
            const canvas = document.getElementById('frontend-intake-age-chart');
            if (!canvas) return;
            
            const ageCounts = {};
            intakes.forEach(intake => {
                const age = intake.age || 'Unknown';
                ageCounts[age] = (ageCounts[age] || 0) + 1;
            });
            
            const colors = ['#ef4444', '#f97316', '#eab308', '#22c55e', '#06b6d4', '#3b82f6', '#8b5cf6', '#ec4899'];
            const chartData = Object.entries(ageCounts).map(([age, count], index) => ({
                label: age,
                data: count,
                backgroundColor: colors[index % colors.length]
            }));
            
            new Chart(canvas.getContext('2d'), {
                type: 'pie',
                data: {
                    labels: chartData.map(d => d.label),
                    datasets: [{
                        data: chartData.map(d => d.data),
                        backgroundColor: chartData.map(d => d.backgroundColor)
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false
                }
            });
        }
        
        function createFrontendDischargeOutcomeChart(discharges) {
            const canvas = document.getElementById('frontend-discharge-outcome-chart');
            if (!canvas) return;
            
            const outcomeCounts = {};
            discharges.forEach(discharge => {
                const outcome = discharge.outcome || 'Unknown';
                outcomeCounts[outcome] = (outcomeCounts[outcome] || 0) + 1;
            });
            
            const colors = ['#ef4444', '#f97316', '#eab308', '#22c55e', '#06b6d4', '#3b82f6', '#8b5cf6', '#ec4899'];
            const chartData = Object.entries(outcomeCounts).map(([outcome, count], index) => ({
                label: outcome,
                data: count,
                backgroundColor: colors[index % colors.length]
            }));
            
            new Chart(canvas.getContext('2d'), {
                type: 'pie',
                data: {
                    labels: chartData.map(d => d.label),
                    datasets: [{
                        data: chartData.map(d => d.data),
                        backgroundColor: chartData.map(d => d.backgroundColor)
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false
                }
            });
        }
        </script>
        
        <?php
        return ob_get_clean();
    }
    
    /**
     * Simple summary shortcode for frontend display
     */
    public function render_summary($atts) {
        // Get custom organization name for default title
        $default_title = get_option('claws_organization_name', 'CLAWS') . ' Summary';
        
        $atts = shortcode_atts(array(
            'style' => 'cards', // 'cards' or 'simple'
            'title' => $default_title,
            'show_title' => 'true'
        ), $atts, 'claws_summary');
        
        // Get data for summary stats
        $colonies = CLAWS_Database::getInstance()->get_colonies();
        $total_cats = array_sum(array_column($colonies, 'cats'));
        $total_intakes = wp_count_posts('claws_intake')->publish;
        $total_discharges = wp_count_posts('claws_discharge')->publish;
        
        ob_start();
        ?>
        <div class="claws-summary-wrapper" style="font-family: system-ui, -apple-system, sans-serif; max-width: 800px; margin: 0 auto;">
            
            <?php if ($atts['show_title'] === 'true'): ?>
            <div style="text-align: center; margin-bottom: 2rem;">
                <h2 style="font-size: 1.75rem; font-weight: bold; color: #1f2937; margin: 0 0 0.5rem;"><?php echo esc_html($atts['title']); ?></h2>
                <p style="color: #6b7280; font-size: 1rem;"><?php echo esc_html(get_option('claws_organization_tagline', 'Compassionate care for every cat')); ?></p>
            </div>
            <?php endif; ?>
            
            <?php if ($atts['style'] === 'cards'): ?>
            <!-- Card Style -->
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(180px, 1fr)); gap: 1rem;">
                <div style="background: linear-gradient(135deg, #3b82f6, #1d4ed8); color: white; padding: 1.5rem; border-radius: 0.75rem; text-align: center; box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);">
                    <div style="font-size: 2rem; font-weight: bold; margin-bottom: 0.5rem;"><?php echo count($colonies); ?></div>
                    <div style="font-size: 0.875rem; opacity: 0.9;">Colonies</div>
                </div>
                <div style="background: linear-gradient(135deg, #ef4444, #dc2626); color: white; padding: 1.5rem; border-radius: 0.75rem; text-align: center; box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);">
                    <div style="font-size: 2rem; font-weight: bold; margin-bottom: 0.5rem;"><?php echo $total_cats; ?></div>
                    <div style="font-size: 0.875rem; opacity: 0.9;">Total Cats</div>
                </div>
                <div style="background: linear-gradient(135deg, #10b981, #059669); color: white; padding: 1.5rem; border-radius: 0.75rem; text-align: center; box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);">
                    <div style="font-size: 2rem; font-weight: bold; margin-bottom: 0.5rem;"><?php echo $total_intakes; ?></div>
                    <div style="font-size: 0.875rem; opacity: 0.9;">Intakes</div>
                </div>
                <div style="background: linear-gradient(135deg, #f59e0b, #d97706); color: white; padding: 1.5rem; border-radius: 0.75rem; text-align: center; box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);">
                    <div style="font-size: 2rem; font-weight: bold; margin-bottom: 0.5rem;"><?php echo $total_discharges; ?></div>
                    <div style="font-size: 0.875rem; opacity: 0.9;">Outcomes</div>
                </div>
            </div>
            <?php else: ?>
            <!-- Simple Style -->
            <div style="background: white; border: 1px solid #e5e7eb; border-radius: 0.5rem; padding: 1.5rem; box-shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.1);">
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(150px, 1fr)); gap: 1rem; text-align: center;">
                    <div>
                        <div style="font-size: 1.5rem; font-weight: bold; color: #3b82f6;"><?php echo count($colonies); ?></div>
                        <div style="font-size: 0.875rem; color: #6b7280;">Colonies</div>
                    </div>
                    <div>
                        <div style="font-size: 1.5rem; font-weight: bold; color: #ef4444;"><?php echo $total_cats; ?></div>
                        <div style="font-size: 0.875rem; color: #6b7280;">Total Cats</div>
                    </div>
                    <div>
                        <div style="font-size: 1.5rem; font-weight: bold; color: #10b981;"><?php echo $total_intakes; ?></div>
                        <div style="font-size: 0.875rem; color: #6b7280;">Intakes</div>
                    </div>
                    <div>
                        <div style="font-size: 1.5rem; font-weight: bold; color: #f59e0b;"><?php echo $total_discharges; ?></div>
                        <div style="font-size: 0.875rem; color: #6b7280;">Outcomes</div>
                    </div>
                </div>
            </div>
            <?php endif; ?>
            
            <div style="text-align: center; padding: 1rem 0; margin-top: 1rem; border-top: 1px solid #e5e7eb;">
                <p style="color: #6b7280; font-size: 0.75rem; margin: 0;">
                    Data updated in real-time • Powered by <?php echo esc_html(get_option('claws_organization_name', 'CLAWS')); ?>
                </p>
            </div>
            
        </div>
        <?php
        return ob_get_clean();
    }
    
    /**
     * Get organization logo URL
     */
    private function get_organization_logo_url() {
        $logo_id = get_option('claws_organization_logo', 0);
        if ($logo_id) {
            $logo_url = wp_get_attachment_url($logo_id);
            return $logo_url ? $logo_url : '';
        }
        return '';
    }
}
?>